from rest_framework import views, status, generics, permissions
from rest_framework.response import Response
from drf_spectacular.utils import extend_schema, OpenApiParameter
from django.core.exceptions import ValidationError

from .models import Wallet, Transaction
from .serializers import WalletSerializer, WithdrawalRequestSerializer, TransactionSerializer
from .services import WalletService

class WalletBalanceView(views.APIView):
    permission_classes = [permissions.IsAuthenticated]

    @extend_schema(
        summary="Get Wallet Balance",
        description="Returns the current user's wallet balance and status.",
        responses={200: WalletSerializer}
    )
    def get(self, request):
        wallet = request.user.wallet
        serializer = WalletSerializer(wallet)
        return Response(serializer.data)

class WithdrawalRequestView(views.APIView):
    permission_classes = [permissions.IsAuthenticated]

    @extend_schema(
        summary="Request Withdrawal",
        description="Deducts balance immediately if funds exist.",
        request=WithdrawalRequestSerializer,
        responses={
            200: WalletSerializer,
            400: dict
        }
    )
    def post(self, request):
        serializer = WithdrawalRequestSerializer(data=request.data)
        if serializer.is_valid():
            amount = serializer.validated_data['amount']
            desc = serializer.validated_data.get('description', 'User Withdrawal')
            
            try:
                # Service handles Locking and Validation
                updated_wallet = WalletService.withdraw(request.user, amount, desc)
                return Response(WalletSerializer(updated_wallet).data, status=status.HTTP_200_OK)
            
            except ValidationError as e:
                return Response({"error": str(e)}, status=status.HTTP_400_BAD_REQUEST)
        
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class TransactionHistoryView(generics.ListAPIView):
    permission_classes = [permissions.IsAuthenticated]
    serializer_class = TransactionSerializer

    @extend_schema(
        summary="Get Transaction History",
        description="Paginated list of user transactions."
    )
    def get_queryset(self):
        return Transaction.objects.filter(wallet__user=self.request.user).order_by('-created_at')