from django.db import transaction
from django.core.exceptions import ValidationError
from decimal import Decimal
from .models import Wallet, Transaction

class WalletService:
    @staticmethod
    def deposit(user, amount: Decimal, description: str = "System Deposit") -> Wallet:
        """
        Increases user balance atomically.
        Used by Data Processor or Admin.
        """
        if amount <= 0:
            raise ValidationError("Deposit amount must be positive.")

        with transaction.atomic():
            # LOCK the row. No other process can write to this wallet until this block finishes.
            wallet = Wallet.objects.select_for_update().get(user=user)
            
            if wallet.is_frozen:
                raise ValidationError("Wallet is frozen. Cannot deposit.")

            wallet.balance += amount
            wallet.save()

            Transaction.objects.create(
                wallet=wallet,
                amount=amount,
                transaction_type=Transaction.TransactionType.DEPOSIT,
                description=description
            )
            return wallet

    @staticmethod
    def withdraw(user, amount: Decimal, description: str = "User Withdrawal") -> Wallet:
        """
        Decreases user balance atomically.
        """
        if amount <= 0:
            raise ValidationError("Withdrawal amount must be positive.")

        with transaction.atomic():
            # LOCK the row.
            wallet = Wallet.objects.select_for_update().get(user=user)

            if wallet.is_frozen:
                raise ValidationError("Wallet is frozen. Cannot withdraw.")

            if wallet.balance < amount:
                raise ValidationError(f"Insufficient funds. Balance: {wallet.balance}")

            wallet.balance -= amount
            wallet.save()

            Transaction.objects.create(
                wallet=wallet,
                amount=amount,
                transaction_type=Transaction.TransactionType.WITHDRAWAL,
                description=description
            )
            return wallet