from django.db import models
from django.conf import settings
from apps.common.models import TimeStampedModel
from django.utils.translation import gettext_lazy as _

class Wallet(TimeStampedModel):
    user = models.OneToOneField(
        settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='wallet'
    )
    balance = models.DecimalField(
        max_digits=14, 
        decimal_places=2, 
        default=0.00,
        verbose_name=_("Current Balance")
    )
    is_frozen = models.BooleanField(
        default=False,
        help_text=_("Freezes the wallet for any outgoing transactions.")
    )

    def __str__(self):
        return f"Wallet ({self.user.username}) - {self.balance}"

class Transaction(TimeStampedModel):
    class TransactionType(models.TextChoices):
        DEPOSIT = 'DEPOSIT', _('Deposit')
        WITHDRAWAL = 'WITHDRAWAL', _('Withdrawal')

    wallet = models.ForeignKey(
        Wallet,
        on_delete=models.PROTECT, # Prevent deleting wallet if it has history
        related_name='transactions'
    )
    amount = models.DecimalField(max_digits=14, decimal_places=2)
    transaction_type = models.CharField(
        max_length=20, 
        choices=TransactionType.choices
    )
    description = models.CharField(max_length=255, blank=True)

    def __str__(self):
        return f"{self.transaction_type} - {self.amount} ({self.wallet.user.username})"