from django.conf import settings
from django.shortcuts import get_object_or_404
from .models import ClickLog
from apps.accounts.models import CustomUser
from apps.campaigns.models import Campaign

class TrackingService:
    """
    Business logic for tracking, deep linking, and redirection.
    """

    @staticmethod
    def generate_affiliate_link(campaign_id: str, affiliate_id: str) -> str:
        """
        Generates the unique deep link for an affiliate.
        Format: https://domain.com/api/v1/tracking/c/{campaign_id}/{affiliate_id}/
        """
        # Ensure site_url does not have a trailing slash
        base_url = settings.SITE_URL.rstrip('/')
        return f"{base_url}/api/v1/tracking/c/{campaign_id}/{affiliate_id}/"

    @staticmethod
    def log_click(campaign_id: str, affiliate_id: str, meta: dict) -> str:
        """
        1. Verifies Campaign and User exist.
        2. Logs the click to DB (Direct Write).
        3. Returns the Target URL for redirection.
        
        *Note: In high-scale scenarios, step 2 is moved to Redis/Celery tasks.*
        """
        # Using select_related/only optimization not strictly necessary here 
        # unless we need specific fields, but standard lookup is fine for MVP.
        campaign = get_object_or_404(Campaign, id=campaign_id)
        affiliate = get_object_or_404(CustomUser, id=affiliate_id)

        # Create the log entry
        ClickLog.objects.create(
            campaign=campaign,
            affiliate=affiliate,
            ip_address=meta.get('ip_address'),
            user_agent=meta.get('user_agent'),
            referer=meta.get('referer')
        )

        return campaign.target_url