from django.db import models
from django.utils.translation import gettext_lazy as _
from apps.common.models import TimeStampedModel, UUIDModel

class ClickLog(UUIDModel, TimeStampedModel):
    """
    High-volume table recording every click.
    """
    # Using string references to avoid circular import issues at module level
    affiliate = models.ForeignKey(
        'accounts.CustomUser', 
        on_delete=models.SET_NULL, 
        null=True, 
        related_name='click_logs',
        help_text="The affiliate who owns the link."
    )
    campaign = models.ForeignKey(
        'campaigns.Campaign', 
        on_delete=models.CASCADE, 
        related_name='click_logs'
    )
    
    # Metadata for fraud detection and analytics
    ip_address = models.GenericIPAddressField(null=True, blank=True)
    user_agent = models.TextField(null=True, blank=True)
    referer = models.URLField(max_length=500, null=True, blank=True)

    class Meta:
        ordering = ['-created_at']
        indexes = [
            models.Index(fields=['created_at']),
            models.Index(fields=['affiliate', 'campaign']),
        ]

    def __str__(self):
        return f"Click: {self.campaign} by {self.affiliate} ({self.created_at})"


class FunnelEvent(UUIDModel, TimeStampedModel):
    """
    Tracks the lifecycle of a user action for attribution.
    """
    class EventType(models.TextChoices):
        CLICK = 'CLICK', _('Click')
        REGISTER = 'REGISTER', _('Registration')
        TRADE = 'TRADE', _('Trade Executed')

    class SourceType(models.TextChoices):
        SYSTEM = 'SYSTEM', _('System Tracking')
        EXCEL = 'EXCEL', _('Excel Import')

    affiliate = models.ForeignKey(
        'accounts.CustomUser', 
        on_delete=models.SET_NULL, 
        null=True
    )
    campaign = models.ForeignKey(
        'campaigns.Campaign', 
        on_delete=models.CASCADE
    )
    event_type = models.CharField(
        max_length=20, 
        choices=EventType.choices, 
        default=EventType.CLICK
    )
    source = models.CharField(
        max_length=20, 
        choices=SourceType.choices, 
        default=SourceType.SYSTEM
    )
    
    # Optional metadata (e.g., specific trade ID or external ID)
    metadata = models.JSONField(default=dict, blank=True)

    def __str__(self):
        return f"{self.event_type} - {self.campaign.title}"