import logging
from aiogram import Router, F, types
from aiogram.filters import CommandStart, CommandObject
from aiogram.fsm.context import FSMContext
from asgiref.sync import sync_to_async

# UI & State Imports
from apps.telegram_bot.keyboards.reply import get_contact_keyboard
from apps.telegram_bot.keyboards.inline import get_main_menu_keyboard
from apps.telegram_bot.states import RegistrationState

# Service Imports
from apps.telegram_bot.services import (
    get_welcome_message,
    get_maintenance_message,
    is_bot_active
)
# We import the registration logic from the Accounts app
# Ensure backend/apps/accounts/services.py has a 'register_user' function
from apps.accounts.services import register_user

logger = logging.getLogger(__name__)
router = Router()

@router.message(CommandStart())
async def bot_start(
    message: types.Message, 
    command: CommandObject, 
    state: FSMContext, 
    user=None
):
    """
    Handles the /start command.
    Payload (e.g., /start 12345) is used for referral tracking.
    """
    
    # 1. Check Maintenance Mode (controlled via Admin Panel)
    if not await is_bot_active():
        maintenance_text = await get_maintenance_message()
        await message.answer(maintenance_text)
        return

    # 2. Extract Referral Code (Deep Link)
    referral_code = command.args  # This captures 'ref_123' from 't.me/bot?start=ref_123'
    
    # 3. Handle Already Registered Users
    if user:
        # If the user exists, we skip registration.
        # Ideally, if there's a referral code, you might want to log it for re-attribution
        # or redirect them to a specific campaign, but standard logic is just showing the menu.
        
        welcome_text = await get_welcome_message(message.from_user.first_name)
        await message.answer(
            f"👋 <b>Welcome back, {user.first_name}!</b>\n\n"
            f"Use the menu below to manage your campaigns.",
            reply_markup=get_main_menu_keyboard(is_affiliate=True),
            parse_mode="HTML"
        )
        return

    # 4. Handle New Users (Registration Flow)
    await state.set_state(RegistrationState.waiting_for_contact)
    
    # Store the referral code in FSM state so we can use it after they share contact
    if referral_code:
        await state.update_data(referral_code=referral_code)
        logger.info(f"New user started with referral code: {referral_code}")

    # Fetch dynamic welcome text from DB
    welcome_text = await get_welcome_message(message.from_user.first_name)
    
    await message.answer(
        f"{welcome_text}\n\n"
        "👇 <b>Action Required:</b>\n"
        "Please tap the button below to securely share your phone number and complete your registration.",
        reply_markup=get_contact_keyboard(),
        parse_mode="HTML"
    )


@router.message(RegistrationState.waiting_for_contact, F.contact)
async def handle_contact_registration(message: types.Message, state: FSMContext):
    """
    Handles the contact shared by the user.
    Calls the Django 'accounts' app to create the user in Postgres.
    """
    contact = message.contact
    
    # Security: Ensure the shared contact belongs to the sender
    if message.from_user.id != contact.user_id:
        await message.answer("❌ Please share your own contact using the button below.")
        return

    # Retrieve referral code from state
    state_data = await state.get_data()
    referral_code = state_data.get("referral_code")

    await message.answer("⏳ Verifying information...")

    try:
        # WRAPPER: Call the Synchronous Django Service
        # We assume register_user returns a Django User instance or None
        user = await sync_to_async(register_user)(
            phone_number=contact.phone_number,
            telegram_id=message.from_user.id,
            first_name=message.from_user.first_name or "",
            last_name=message.from_user.last_name or "",
            username=message.from_user.username or "",
            referral_code=referral_code
        )

        # Clear FSM State
        await state.clear()

        if user:
            await message.answer(
                "✅ <b>Registration Successful!</b>\n\n"
                "You are now an active Affiliate on TablighShow.\n"
                "Use the menu below to start earning!",
                reply_markup=get_main_menu_keyboard(is_affiliate=True),
                parse_mode="HTML"
            )
        else:
            # Fallback if service returned None without raising exception
            await message.answer(
                "⚠️ Registration could not be completed. Please contact support."
            )

    except Exception as e:
        logger.error(f"Registration Error for ID {message.from_user.id}: {e}", exc_info=True)
        await message.answer(
            "❌ <b>System Error</b>\n\n"
            "We encountered an issue registering your account. "
            "Please try again later or contact admin.",
            parse_mode="HTML"
        )