import pandas as pd
import logging
from django.db import transaction
from django.core.exceptions import ObjectDoesNotExist

# These imports are safe assuming models are defined
from apps.data_processor.models import ImportJob
from apps.accounts.models import CustomUser
from apps.wallet.services import deposit 

logger = logging.getLogger(__name__)

    # ... (rest of the class code)
class ExcelProcessor:
    """
    Handles reading Excel files, normalizing data, and executing logic
    for finding users and rewarding referrers.
    """

    REQUIRED_COLUMNS = ['phone_number']

    def __init__(self, job_id):
        self.job = ImportJob.objects.get(id=job_id)
        self.log_summary = {
            "total_rows": 0,
            "processed": 0,
            "skipped": 0,
            "errors": []
        }

    def run(self):
        self.job.status = ImportJob.Status.PROCESSING
        self.job.save()

        try:
            self._process_file()
            self.job.status = ImportJob.Status.COMPLETED
        except Exception as e:
            logger.exception(f"Import Job {self.job.id} Failed")
            self.job.status = ImportJob.Status.FAILED
            self.log_summary['errors'].append(str(e))
        finally:
            self.job.processing_log = self.log_summary
            self.job.save()

    def _process_file(self):
        file_path = self.job.file.path
        
        # 1. Read Excel using Pandas
        try:
            df = pd.read_excel(file_path)
        except Exception as e:
            raise ValueError(f"Failed to read Excel file: {str(e)}")

        # 2. Validate Headers
        missing_cols = [col for col in self.REQUIRED_COLUMNS if col not in df.columns]
        if missing_cols:
            raise ValueError(f"Missing required columns: {missing_cols}")

        self.log_summary['total_rows'] = len(df)
        campaign_reward = self.job.campaign.payout_amount  # Assuming Campaign has this field

        # 3. Iterate Rows
        for index, row in df.iterrows():
            try:
                raw_phone = str(row['phone_number'])
                clean_phone = self._normalize_phone(raw_phone)

                self._handle_row(clean_phone, campaign_reward)
                self.log_summary['processed'] += 1

            except Exception as e:
                self.log_summary['skipped'] += 1
                self.log_summary['errors'].append(f"Row {index + 2}: {str(e)}")

    def _handle_row(self, phone, reward_amount):
        # 1. Find the User who registered on the exchange
        try:
            user = CustomUser.objects.get(phone_number=phone)
        except CustomUser.DoesNotExist:
            raise ObjectDoesNotExist(f"User with phone {phone} not found in DB.")

        # 2. Identify who invited them (Referrer)
        referrer = user.invited_by  # Assuming 'invited_by' FK exists on CustomUser
        if not referrer:
            # Valid user, but nobody invited them -> No commission
            return 

        # 3. Process Reward via Wallet App
        # Using atomic transaction ensures we don't double pay if things crash midway
        with transaction.atomic():
            description = f"Commission for referring user {phone} (Campaign: {self.job.campaign.title})"
            
            # Call Wallet Service
            deposit(
                user=referrer,
                amount=reward_amount,
                transaction_type='COMMISSION',
                description=description
            )

    def _normalize_phone(self, phone):
        """
        Basic normalization. Adapts +98, 0098, or 09... to standard format.
        """
        phone = phone.strip()
        if phone.endswith('.0'): # Excel float artifact
            phone = phone[:-2]
            
        # Example logic for Iran phones, adjust as per requirements
        if phone.startswith('+98'):
            phone = '0' + phone[3:]
        elif phone.startswith('98'):
            phone = '0' + phone[2:]
            
        return phone