# backend/apps/campaigns/models.py
from django.db import models
from django.utils.translation import gettext_lazy as _
from apps.common.models import TimeStampedModel, UUIDModel

class Category(TimeStampedModel, UUIDModel):
    name = models.CharField(_("Name"), max_length=100)
    slug = models.SlugField(_("Slug"), unique=True)

    class Meta:
        verbose_name = _("Category")
        verbose_name_plural = _("Categories")

    def __str__(self):
        return self.name


class Campaign(TimeStampedModel, UUIDModel):
    category = models.ForeignKey(
        Category, 
        on_delete=models.SET_NULL, 
        null=True, 
        related_name="campaigns",
        verbose_name=_("Category")
    )
    title = models.CharField(_("Title"), max_length=255)
    slug = models.SlugField(_("Slug"), unique=True)
    description = models.TextField(_("Description"), blank=True)
    target_url = models.URLField(_("Target URL"), help_text=_("Where the user lands."))
    
    # Economics
    cpa_amount = models.DecimalField(
        _("CPA Amount"), 
        max_digits=10, 
        decimal_places=2, 
        help_text=_("Amount paid to affiliate per conversion.")
    )
    
    # Assets
    cover_image = models.ImageField(
        _("Cover Image"), 
        upload_to='campaigns/covers/',
        blank=True, null=True
    )
    is_active = models.BooleanField(_("Is Active"), default=True)

    class Meta:
        verbose_name = _("Campaign")
        verbose_name_plural = _("Campaigns")

    def __str__(self):
        return self.title


class MediaFile(TimeStampedModel, UUIDModel):
    class MediaType(models.TextChoices):
        IMAGE = 'IMAGE', _('Image')
        VIDEO = 'VIDEO', _('Video')

    campaign = models.ForeignKey(
        Campaign, 
        on_delete=models.CASCADE, 
        related_name='media_files'
    )
    file = models.FileField(_("File"), upload_to='campaigns/assets/')
    file_type = models.CharField(
        _("Type"), 
        max_length=10, 
        choices=MediaType.choices, 
        default=MediaType.IMAGE
    )

    class Meta:
        verbose_name = _("Media File")
        verbose_name_plural = _("Media Files")
    
    def __str__(self):
        return f"{self.campaign.title} - {self.file_type}"