from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from rest_framework import status
from drf_spectacular.utils import extend_schema

from .services import OTPService, AuthService
from .serializers import (
    SendOTPSerializer, 
    VerifyOTPSerializer, 
    AuthResponseSerializer,
    UserSerializer
)

class SendOTPView(APIView):
    permission_classes = []

    @extend_schema(
        request=SendOTPSerializer,
        responses={200: {"detail": "OTP sent"}},
        summary="Send Login OTP",
        description="Generates an OTP code and prints it to the console (Simulated SMS)."
    )
    def post(self, request):
        serializer = SendOTPSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        
        phone = serializer.validated_data['phone_number']
        OTPService.generate_otp(phone)
        
        return Response({"detail": "OTP sent to console."}, status=status.HTTP_200_OK)


class VerifyOTPView(APIView):
    permission_classes = []

    @extend_schema(
        request=VerifyOTPSerializer,
        responses={200: AuthResponseSerializer},
        summary="Verify OTP & Login",
        description="Validates OTP. Creates user if not exists. Returns JWT tokens."
    )
    def post(self, request):
        serializer = VerifyOTPSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        
        phone = serializer.validated_data['phone_number']
        code = serializer.validated_data['code']
        referral_code = serializer.validated_data.get('referral_code')

        if not OTPService.validate_otp(phone, code):
            return Response(
                {"detail": "Invalid or expired OTP."}, 
                status=status.HTTP_400_BAD_REQUEST
            )

        auth_data = AuthService.authenticate_or_register(phone, referral_code)
        
        # Serialize response
        response_data = {
            'refresh': auth_data['tokens']['refresh'],
            'access': auth_data['tokens']['access'],
            'user': UserSerializer(auth_data['user']).data
        }
        
        return Response(response_data, status=status.HTTP_200_OK)


class MeView(APIView):
    permission_classes = [IsAuthenticated]

    @extend_schema(
        responses={200: UserSerializer},
        summary="Get Current User Profile",
        description="Returns the profile details of the currently authenticated user."
    )
    def get(self, request):
        serializer = UserSerializer(request.user)
        return Response(serializer.data)